using inetsoftware.Pdfc;
using inetsoftware.Pdfc.Generator.Message;
using inetsoftware.Pdfc.Generator.Model;
using inetsoftware.Pdfc.Results;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Collections.Generic;

namespace inetsoftware.PdfcSamples
{
    /// <summary>
    /// A sample for writing JUnit test cases using PDFC
    /// </summary>
    [TestClass]
    public class CompareTest
    {
        static CompareTest()
        {
            // Activate the license once from environment if required.
            string key = Environment.GetEnvironmentVariable("PDFC_KEY");
            if (!string.IsNullOrEmpty(key))
            {
                PDFC.ActivateLicense(key);
            }
        }

        private static PDFComparer pdfComparer;

        /// <summary>
        /// Checks for an valid, active license and request a trial license, if there is none.
        /// <para>
        /// Will also initialize the PDFCComparer.
        /// </para>
        /// </summary>        
        [ClassInitialize]
        public static void Before()
        {
            pdfComparer = new PDFComparer();
        }

        /// <summary>
        /// Checks two PDF files for differences.
        /// </summary>
        [TestMethod]
        public void TestDifferencesCount()
        {
            string example1 = "samples/resources/Example1.pdf";
            string example2 = "samples/resources/Example2.pdf";

            using (ResultModel result = pdfComparer.Compare(example1, example2))
            {
                InfoData comparisonParameters = result.GetComparisonParameters();

                Assert.AreEqual(11, result.GetDifferencesCount(false));
                Assert.AreEqual(11, result.GetDifferencesCount(true));

                Assert.AreEqual(3, comparisonParameters.FirstPageCount);
                Assert.AreEqual(3, comparisonParameters.SecondPageCount);
            }
        }
    

        /// <summary>
        /// Compares two PDF files and checks the DiffGroups.
        /// </summary>
        [TestMethod]
        public void TestDifferenceGroupsList()
        {
            string example1 = "samples/resources/Example1.pdf";
            string example2 = "samples/resources/Example2.pdf";

            using (ResultModel result = pdfComparer.Compare(example1, example2))
            {
                List<DiffGroups> diffsFileA = result.GetDifferenceGroupsList(true);
                List<DiffGroups> diffsFileB = result.GetDifferenceGroupsList(false);
                
                Assert.AreEqual(9, diffsFileA.Count);
                Assert.AreEqual(9, diffsFileB.Count);

                DiffGroups groupForCheck = null;

                foreach(DiffGroups group in diffsFileA)
                {
                    if(group.Message.StartsWith("AddedOrRemoved"))
                    {
                        groupForCheck = group;
                        break;
                    }
                }

                Assert.IsNotNull(groupForCheck);
                // if a group was added or removed, the bounds must be different
                Assert.AreNotEqual(groupForCheck.BoundsFirst, groupForCheck.BoundsSecond);
            }
        }

        /// <summary>
        /// Compares two PDF files and checks a single difference.
        /// </summary>
        [TestMethod]
        public void TestDifferences()
        {
            string example1 = "samples/resources/Example1.pdf";
            string example2 = "samples/resources/Example2.pdf";

            using (ResultModel result = pdfComparer.Compare(example1, example2))
            {

                List<Diff> diffsFileA = result.GetDifferences(true);
                List<Diff> diffsFileB = result.GetDifferences(false);

                Diff diffToCheck = null;

                foreach (Diff diff in diffsFileA)
                {

                    if(diff.Type == "Modified")
                    {
                        diffToCheck = diff;
                        break;
                    }
                }

                Assert.IsNotNull(diffToCheck);
                // the element should be in the modified message
                Assert.IsTrue(diffToCheck.Message.Contains("'" + diffToCheck.Element + "'"));

            }
        }
    }

}